# routes.py
from flask import jsonify, request, session
import struct
import os
import hashlib
from pathlib import Path
import threading
import secrets
from time import sleep, time

REQUIRE_INVITE = True
MAX_ACCOUNTS = 64

def register_routes(app, send_udp_command, require_auth, require_admin, 
                   create_ssl_connection, GLOBAL_SECRET, DOMAIN,
                   users, save_users, spawn_resilience, remove_resilience,
                   payment_requests, payment_lock, add_payment_request):

    INVITE_COUNTER_FILE = Path("invite_counter.txt")
    
    user_lock = threading.Lock()
    login_lock = threading.Lock()

    def get_invite_counter():
        if INVITE_COUNTER_FILE.exists():
            with open(INVITE_COUNTER_FILE, 'rb') as f:
                return struct.unpack('>I', f.read(4))[0]
        return 0

    def set_invite_counter(value):
        with open(INVITE_COUNTER_FILE, 'wb') as f:
            f.write(struct.pack('>I', value))

    def generate_invite_code(counter):
        data = counter.to_bytes(4, 'big') + GLOBAL_SECRET
        return hashlib.sha256(data).hexdigest()[:12]

    # ===== USER ROUTES =====

    @app.route('/api/user/trust_index', methods=['POST'])
    @require_auth
    def set_trust_index():
        try:
            args = struct.pack('>f', float(request.json.get('value')))
            status, data = send_udp_command(session['user_id'], 0, args)
        except:
            return jsonify({'error': 'Invalid value'}), 400
        if status != 0:
            return jsonify({'error': data.decode('utf-8', errors='ignore')}), 500
        return jsonify({'message': data.decode('utf-8', errors='ignore')})

    @app.route('/api/user/add_account', methods=['POST'])
    @require_auth
    def add_account():
        try:
            args = (
                request.json.get('username', '').encode().ljust(32, b'\0')[:32] +
                request.json.get('server', '').encode().ljust(32, b'\0')[:32] +
                struct.pack('>H', request.json.get('port', 0)) +
                bytes.fromhex(request.json.get('key', ''))
            )
            status, response = send_udp_command(session['user_id'], 1, args)
        except:
            return jsonify({'error': 'Invalid input'}), 400
        
        if status != 0:
            return jsonify({'error': response.decode('utf-8', errors='ignore')}), 500
        return jsonify({'message': response.decode('utf-8', errors='ignore')})

    @app.route('/api/user/remove_account', methods=['POST'])
    @require_auth
    def remove_account():
        try:
            args = (
                request.json.get('username', '').encode().ljust(32, b'\0')[:32] +
                request.json.get('server', '').encode().ljust(32, b'\0')[:32]
            )
            status, response = send_udp_command(session['user_id'], 2, args)
        except:
            return jsonify({'error': 'Invalid input'}), 400
        
        if status != 0:
            return jsonify({'error': response.decode('utf-8', errors='ignore')}), 500
        return jsonify({'message': response.decode('utf-8', errors='ignore')})

    @app.route('/api/user/set_trustline', methods=['POST'])
    @require_auth
    def set_trustline():
        try:
            args = (
                request.json.get('username', '').encode().ljust(32, b'\0')[:32] +
                request.json.get('server', '').encode().ljust(32, b'\0')[:32] +
                struct.pack('>Q', request.json.get('amount', 0))
            )
            status, response = send_udp_command(session['user_id'], 3, args)
        except:
            return jsonify({'error': 'Invalid input'}), 400
        
        if status != 0:
            return jsonify({'error': response.decode('utf-8', errors='ignore')}), 500
        return jsonify({'message': response.decode('utf-8', errors='ignore')})

    @app.route('/api/user/payment', methods=['POST'])
    @require_auth
    def payment():
        try:
            args = (
                request.json.get('username', '').encode().ljust(32, b'\0')[:32] +
                request.json.get('server', '').encode().ljust(32, b'\0')[:32] +
                struct.pack('>H', request.json.get('port', 0)) +
                bytes.fromhex(request.json.get('key', '')) +
                struct.pack('>Q', request.json.get('amount', 0))
            )
            status, response = send_udp_command(session['user_id'], 4, args)
        except:
            return jsonify({'error': 'Invalid input'}), 400
        
        if status != 0:
            return jsonify({'error': response.decode('utf-8', errors='ignore')}), 500
        return jsonify({'message': response.decode('utf-8', errors='ignore')})

    @app.route('/api/user/receive_payment', methods=['POST'])
    @require_auth
    def receive_payment():
        try:
            args = (
                request.json.get('username', '').encode().ljust(32, b'\0')[:32] +
                request.json.get('server', '').encode().ljust(32, b'\0')[:32] +
                struct.pack('>H', request.json.get('port', 0)) +
                bytes.fromhex(request.json.get('key', ''))
            )
            status, response = send_udp_command(session['user_id'], 5, args)
        except:
            return jsonify({'error': 'Invalid input'}), 400
        
        if status != 0:
            return jsonify({'error': response.decode('utf-8', errors='ignore')}), 500
        return jsonify({'message': response.decode('utf-8', errors='ignore')})

    @app.route('/api/user/local_payment', methods=['POST'])
    @require_auth
    def local_payment():
        try:
            args = (
                request.json.get('username', '').encode().ljust(32, b'\0')[:32] +
                request.json.get('server', '').encode().ljust(32, b'\0')[:32] +
                struct.pack('>Q', request.json.get('amount', 0))
            )
            status, response = send_udp_command(session['user_id'], 6, args)
        except:
            return jsonify({'error': 'Invalid input'}), 400
        
        if status != 0:
            return jsonify({'error': response.decode('utf-8', errors='ignore')}), 500
        return jsonify({'message': response.decode('utf-8', errors='ignore')})

    @app.route('/api/user/trust_index', methods=['GET'])
    @require_auth
    def get_trust_index():
        status, data = send_udp_command(session['user_id'], 7, b'')
        if status != 0:
            return jsonify({'error': data.decode('utf-8', errors='ignore')}), 500
        return jsonify({'value': struct.unpack('>f', data[:4])[0]})
    
    @app.route('/api/user/accounts', methods=['GET'])
    @require_auth
    def get_accounts():
        status, data = send_udp_command(session['user_id'], 8, b'')
        if status != 0:
            return jsonify({'error': data.decode('utf-8', errors='ignore')}), 500
        
        count = data[0]
        accounts = []
        
        for i in range(count):
            status, data = send_udp_command(session['user_id'], 9, bytes([i]))
            if status == 0 and len(data) >= 88:
                accounts.append({
                    'username': data[0:32].decode('utf-8', errors='ignore').split('\0')[0],
                    'server': data[32:64].decode('utf-8', errors='ignore').split('\0')[0],
                    'balance': struct.unpack('>q', data[64:72])[0],
                    'limit_out': struct.unpack('>Q', data[72:80])[0],
                    'limit_in': struct.unpack('>Q', data[80:88])[0]
                })
        
        return jsonify(accounts)

    @app.route('/api/user/payments', methods=['GET'])
    @require_auth
    def get_payments():
        payments = []
        
        for i in range(16):
            status, data = send_udp_command(session['user_id'], 10, bytes([i]))
            if status != 0:
                continue
                
            phase = data[0]
            if phase == 0:
                continue
                
            payments.append({
                'id': data[1:33].hex(),
                'recipient': data[33:65].decode('utf-8', errors='ignore').split('\0')[0],
                'server': data[65:97].decode('utf-8', errors='ignore').split('\0')[0],
                'amount': struct.unpack('>Q', data[97:105])[0],
                'direction': 'outgoing' if data[105] == 1 else 'incoming',
                'phase': phase,
                'time_val': struct.unpack('>q', data[106:114])[0]
            })
        
        return jsonify(payments)

    @app.route('/api/user/receipts', methods=['GET'])
    @require_auth
    def get_receipts():
        receipts = []
        
        for i in range(16):
            status, data = send_udp_command(session['user_id'], 11, bytes([i]))
            if status != 0:
                break
            
            receipts.append({
                'type': data[0],
                'id': data[1:33].hex(),
                'username': data[33:65].decode('utf-8', errors='ignore').split('\0')[0],
                'server': data[65:97].decode('utf-8', errors='ignore').split('\0')[0],
                'amount': struct.unpack('>q', data[97:105])[0],
                'timestamp': struct.unpack('>q', data[105:113])[0],
                'status': data[113]
            })
        
        return jsonify(sorted(
            [r for r in receipts if r['timestamp'] > 0],
            key=lambda r: r['timestamp'],
            reverse=True
        ))

    @app.route('/api/user/payment_requests', methods=['GET'])
    @require_auth
    def get_payment_requests():
        with payment_lock:
            current_time = time()
            requests = []
            for req in payment_requests.get(session['user_id'], []):
                if current_time - req['timestamp'] < 900:  # 15 minuter
                    requests.append({
                        'id': req['id'],
                        'sender': req['sender'],
                        'sender_domain': req['sender_domain'],
                        'expires_in': int(900 - (current_time - req['timestamp']))
                    })
            return jsonify(requests)

    @app.route('/api/user/accept_payment', methods=['POST'])
    @require_auth
    def accept_payment():
        request_id = request.json.get('id')
        
        with payment_lock:
            user_requests = payment_requests.get(session['user_id'], [])
            req = None
            for r in user_requests:
                if r['id'] == request_id:
                    req = r
                    user_requests.remove(r)
                    break
        
        if not req:
            return jsonify({'error': 'Request not found or expired'}), 404
        
        args = (
            req['sender'].encode().ljust(32, b'\0')[:32] +
            req['sender_domain'].encode().ljust(32, b'\0')[:32] +
            struct.pack('>H', 2012) +
            req['key']
        )
        status, response = send_udp_command(session['user_id'], 5, args)
        
        if status != 0:
            return jsonify({'error': response.decode('utf-8', errors='ignore')}), 500
        
        return jsonify({'message': 'Payment accepted'})

    @app.route('/api/user/reject_payment', methods=['POST'])
    @require_auth  
    def reject_payment():
        request_id = request.json.get('id')
        
        with payment_lock:
            user_requests = payment_requests.get(session['user_id'], [])
            for r in user_requests:
                if r['id'] == request_id:
                    user_requests.remove(r)
                    return jsonify({'message': 'Request rejected'})
        
        return jsonify({'error': 'Request not found'}), 404


    # ===== FEDERATED ROUTES =====

    @app.route('/api/federated/add_account', methods=['POST'])
    @require_auth
    def federated_add_account():
        username = request.json.get('username', '')
        server = request.json.get('server', '')
        
        try:
            if server == DOMAIN:
                key = GLOBAL_SECRET
            else:
                ssl_sock = create_ssl_connection(server, 8080)
                
                client_domain = DOMAIN.encode().ljust(32, b'\0')[:32]
                client_hash = hashlib.sha256(GLOBAL_SECRET + server.encode()).digest()
                
                # Send msg_type 0 for ADD_ACCOUNT
                ssl_sock.send(b'\x00' + client_domain + client_hash)
                key = ssl_sock.recv(32)
                ssl_sock.close()
            
            args = (
                username.encode().ljust(32, b'\0')[:32] +
                server.encode().ljust(32, b'\0')[:32] +
                struct.pack('>H', 2012) +
                key
            )
            status, response = send_udp_command(session['user_id'], 1, args)
            
        except Exception as e:
            return jsonify({'error': str(e)}), 500
        
        if status != 0:
            return jsonify({'error': response.decode('utf-8', errors='ignore')}), 500
        return jsonify({'message': response.decode('utf-8', errors='ignore')})

    @app.route('/api/federated/payment', methods=['POST'])
    @require_auth
    def federated_payment():
        recipient_username = request.json.get('username', '')
        recipient_server = request.json.get('server', '')
        amount = request.json.get('amount', 0)
        
        try:
            counterpart_key = os.urandom(32)
            
            if recipient_server == DOMAIN:
                if recipient_username not in users:
                    return jsonify({'error': 'User not found'}), 404
                
                if not add_payment_request(recipient_username, session['user_id'], DOMAIN, counterpart_key):
                    return jsonify({'error': 'Too many pending requests'}), 500
            else:
                ssl_sock = create_ssl_connection(recipient_server, 8080)
                
                payment_data = (
                    recipient_username.encode().ljust(32, b'\0')[:32] +
                    session['user_id'].encode().ljust(32, b'\0')[:32] +
                    DOMAIN.encode().ljust(32, b'\0')[:32] +
                    counterpart_key
                )
                # Send msg_type 1 for PAYMENT
                ssl_sock.send(b'\x01' + payment_data)
                response = ssl_sock.recv(16)
                ssl_sock.close()
                
                if response != b'PAYMENT_ACK':
                    return jsonify({'error': 'Payment rejected by recipient'}), 500
            
            args = (
                recipient_username.encode().ljust(32, b'\0')[:32] +
                recipient_server.encode().ljust(32, b'\0')[:32] +
                struct.pack('>H', 2012) +
                counterpart_key +
                struct.pack('>Q', amount)
            )
            status, response = send_udp_command(session['user_id'], 4, args)
            
            if status != 0:
                return jsonify({'error': response.decode('utf-8', errors='ignore')}), 500
            
            return jsonify({'message': 'Payment request sent successfully'})
            
        except Exception as e:
            return jsonify({'error': str(e)}), 500
            
    # ===== AUTH ROUTES =====

    @app.route('/api/auth/login', methods=['POST'])
    def login():
        username = request.json.get('username', '')
        password = request.json.get('password', '')
        
        if username in users:
            user = users[username]
            if secrets.compare_digest(user['password'], password):
                session['user_id'] = username
                session['is_admin'] = user.get('is_admin', False)
                
                return jsonify({
                    'user_id': username,
                    'is_admin': user.get('is_admin', False),
                    'domain': DOMAIN
                })
        
        with login_lock:
            sleep(0.5)
            return jsonify({'error': 'Invalid credentials'}), 401

    @app.route('/api/auth/logout', methods=['POST'])
    def logout():
        session.clear()
        return jsonify({'message': 'Logged out'})

    @app.route('/api/auth/whoami', methods=['GET'])
    @require_auth
    def whoami():
        return jsonify({
            'user_id': session['user_id'],
            'is_admin': session.get('is_admin', False)
        })

    @app.route('/api/auth/change_password', methods=['POST'])
    @require_auth
    def change_password():
        old_password = request.json.get('old_password', '')
        new_password = request.json.get('new_password', '')
        
        with user_lock:
            user = users[session['user_id']]
            if not secrets.compare_digest(user['password'], old_password):
                return jsonify({'error': 'Invalid old password'}), 401
            
            user['password'] = new_password
            save_users()
        
        return jsonify({'message': 'Password changed'})

    # ===== ADMIN ROUTES =====

    @app.route('/api/admin/create_user', methods=['POST'])
    @require_admin
    def create_user():
        user_id = request.json.get('user_id', '')
        password = request.json.get('password', '')
        
        with user_lock:
            if user_id in users:
                return jsonify({'error': 'User already exists'}), 400
            
            users[user_id] = {
                'password': password,
                'is_admin': False
            }
            save_users()
        
        spawn_resilience(user_id)

        return jsonify({'message': 'User created'})

    @app.route('/api/admin/delete_user', methods=['POST'])
    @require_admin
    def delete_user():
        user_id = request.json.get('user_id', '')
        
        with user_lock:
            if user_id not in users:
                return jsonify({'error': 'User not found'}), 404
            
            if users[user_id].get('is_admin', False):
                return jsonify({'error': 'Cannot delete admin user'}), 403
            
            del users[user_id]
            save_users()
        
        remove_resilience(user_id)

        return jsonify({'message': 'User deleted'})

    @app.route('/api/admin/reset_password', methods=['POST'])
    @require_admin
    def reset_password():
        user_id = request.json.get('user_id', '')
        new_password = request.json.get('new_password', '')
        
        with user_lock:
            if user_id not in users:
                return jsonify({'error': 'User not found'}), 404
            
            users[user_id]['password'] = new_password
            save_users()
        
        return jsonify({'message': 'Password reset'})

    @app.route('/api/admin/list_users', methods=['GET'])
    @require_admin
    def list_users():
        return jsonify(list(users.keys()))

    @app.route('/api/admin/create_invite', methods=['POST'])
    @require_admin
    def create_invite():
        counter = get_invite_counter()
        invite = generate_invite_code(counter)
        return jsonify({'invite': invite})

    # ===== INVITE ROUTES =====
    
    @app.route('/api/invite/register', methods=['POST'])
    def register_with_invite():
        if len(users) >= MAX_ACCOUNTS:
            return jsonify({'error': 'Server full - maximum accounts reached'}), 403
        
        current_counter = get_invite_counter()

        if REQUIRE_INVITE and request.json.get('invite') != generate_invite_code(current_counter):
            return jsonify({'error': 'Invalid invite'}), 404
        
        username = request.json.get('username', '').strip()
        password = request.json.get('password', '')
        
        try:
            trust_index = float(request.json.get('trust_index', 0))
            if trust_index < 0 or trust_index > 1:
                trust_index = 0
        except (TypeError, ValueError):
            trust_index = 0
        
        if not username.isalpha():
            return jsonify({'error': 'Username must contain only letters'}), 400

        username = username[:32]

        with user_lock:
            if username in users:
                return jsonify({'error': 'Username already taken'}), 400
            
            users[username] = {'password': password, 'is_admin': False}
            save_users()
        
        spawn_resilience(username)
        
        try:
            args = struct.pack('>f', trust_index)
            send_udp_command(username, 0, args)
        except:
            pass
        
        set_invite_counter(current_counter + 1)
        
        return jsonify({'message': 'User created successfully'})
